#include "StdAfx.h"
#include "VCamSDKImpl.h"

// DirectShow BaseClasses
#ifdef _DEBUG
#pragma comment(lib, "STRMBASD.LIB")
#else
#pragma comment(lib, "STRMBASE.LIB")
#endif

#define SAFE_RELEASE(x) { if (x) x->Release(); x = NULL; }

// VCam filter interface
#pragma warning(disable : 4192)
#import "VCamSource.tlb" no_namespace, raw_interfaces_only exclude("UINT_PTR") 

#define MIDL_DEFINE_GUID(type,name,l,w1,w2,b1,b2,b3,b4,b5,b6,b7,b8) \
	const type name = {l,w1,w2,{b1,b2,b3,b4,b5,b6,b7,b8}}
MIDL_DEFINE_GUID(IID, IID_IVCamSource,0xF8AF4078,0xA31A,0x4651,0x98,0x35,0x47,0xE5,0x4E,0x41,0x52,0x0E);
MIDL_DEFINE_GUID(IID, IID_IVCamRenderer,0xCCFBFF50,0xED42,0x4ce4,0x8E,0x89,0x73,0x24,0xFB,0x08,0x27,0x11);
MIDL_DEFINE_GUID(IID, LIBID_VCamSourceLib,0x54781EAD,0x0C50,0x403a,0xBF,0x3C,0xB1,0x7E,0x24,0x6F,0xE0,0x9F);
MIDL_DEFINE_GUID(CLSID, CLSID_VCamSource,0x1A5C04B6,0x140D,0x4c05,0x99,0x02,0x2E,0x17,0x00,0x17,0x1F,0x3E);
MIDL_DEFINE_GUID(CLSID, CLSID_VCamRenderer,0xB6A665C1,0x955A,0x4d53,0xBF,0x7B,0xA2,0x6D,0x9D,0x4D,0xC2,0x7A);

// Thread used to check filter graph status (whether stopped...)
DWORD WINAPI GraphCheckThread(LPVOID lpParam)
{
	CVCamSDKImpl* pVCam = (CVCamSDKImpl*)lpParam;
	if (pVCam && NULL != pVCam->m_pGraph) {
		IMediaEventEx* pEvent = NULL;
		HRESULT hr = pVCam->m_pGraph->QueryInterface(IID_IMediaEventEx, (void **)&pEvent);

		HANDLE  hEvent; 
		long    evCode, param1, param2;
		BOOLEAN bDone = FALSE;
		hr = pEvent->GetEventHandle((OAEVENT*)&hEvent);
		if (FAILED(hr))
		{
			/* Insert failure-handling code here. */
			SAFE_RELEASE(pEvent);
			return 2;
		}
		while(!bDone && !pVCam->m_bStopGraphCheck && NULL != pVCam->m_pGraph) 
		{
			if (WAIT_OBJECT_0 == WaitForSingleObject(hEvent, 100))
			{ 
				while (hr = pEvent->GetEvent(&evCode, &param1, &param2, 0), SUCCEEDED(hr)) 
				{
					pEvent->FreeEventParams(evCode, param1, param2);
					bDone = (EC_COMPLETE == evCode || EC_USERABORT == evCode);
					ATLTRACE(_T("evCode = 0x%08X\n"), evCode);
				}
			}
		}
		SAFE_RELEASE(pEvent);
		if (bDone) {
			pVCam->StopPlay(); // play to end, ok, let's stop the graph here!
		}
		return 0;
	}
	return 1;
}

//////////////////////////////////////////////////////////////////////////


CVCamSDKImpl::CVCamSDKImpl(void)
	: m_pIVCamSource(NULL)
	, m_hGraphCheckThread(NULL)
	, m_bStopGraphCheck(TRUE)
{
	// ::CoInitialize(NULL);

	HRESULT hr = CoCreateInstance(CLSID_VCamRenderer, NULL, CLSCTX_INPROC, IID_IBaseFilter, 
		reinterpret_cast<void**>(&m_pVCamRenderer));
	if (FAILED(hr)) {
		MessageBox(NULL, _T("Failed to create VCam Renderer filter!"), _T("Error"), MB_OK | MB_ICONERROR);
	}
	m_pVCamRenderer->QueryInterface(IID_IVCamRenderer, reinterpret_cast<void**>(&m_pIVCamRenderer));

	hr = CoCreateInstance(CLSID_VCamSource, NULL, CLSCTX_INPROC, IID_IVCamSource, 
		reinterpret_cast<void**>(&m_pIVCamSource));
	if (FAILED(hr)) {
		MessageBox(NULL, _T("Failed to create VCam Source filter!"), _T("Error"), MB_OK | MB_ICONERROR);
	}
	ATLASSERT(SUCCEEDED(hr));
}

CVCamSDKImpl::~CVCamSDKImpl(void)
{
	StopPlay();

	SAFE_RELEASE(m_pIVCamSource);
	// ::CoUninitialize();
}

void	CVCamSDKImpl::SetLicenseCode(BSTR pRegStr)
{
	if (m_pIVCamSource) {
		m_pIVCamSource->SetLicenseCode(pRegStr);
	}
}

void CVCamSDKImpl::SetProp(ULONG lIndex, long lValue)
{
	if (m_pIVCamSource) {
		m_pIVCamSource->SetProp(lIndex, lValue);
	}
}

//////////////////////////////////////////////////////////////////////////
// DShow Helper functions

// DisconnectFilter - Disconnect all pins on filter.
static void DisconnectFilter(IGraphBuilder *pGraph, IBaseFilter *pFilter)
{
	IEnumPins* pPins;
	IPin* pPin;
	IPin* pConnectedPin;
	DWORD	numPinsFetched;

	HRESULT hr = pFilter->EnumPins(&pPins);
	if (FAILED(hr))
		return;
	pPins->Reset();

	while( SUCCEEDED(pPins->Next(1, &pPin, &numPinsFetched)) && (numPinsFetched == 1) )	{					
		if ( pPin != NULL) {
			pPin->ConnectedTo(&pConnectedPin);
			if (pConnectedPin) {
				pGraph->Disconnect(pPin);
				pGraph->Disconnect(pConnectedPin);
				SAFE_RELEASE(pConnectedPin);
			}
			SAFE_RELEASE(pPin);
		}
	}
	SAFE_RELEASE(pPins);
}

// Disconnect and remove all filters in graph
static HRESULT	TearDownGraph(IGraphBuilder *pGraph)
{
	HRESULT         hr              = S_OK;

	IBaseFilter     *pFilter        = NULL;
	IEnumFilters    *pIFilterEnum   = NULL;


	if (pGraph == NULL) return S_OK;

	// now go unload rendered filters
	hr = pGraph->EnumFilters(&pIFilterEnum);

	if (FAILED(hr)) {
		goto err;
	}

	pIFilterEnum->Reset();

	while ((pIFilterEnum->Next(1, &pFilter, 0) == S_OK) && (pFilter != 0)) {
		DisconnectFilter(pGraph, pFilter);
		if ((hr = pGraph->RemoveFilter(pFilter)) != S_OK) {
			goto err;
		}

		SAFE_RELEASE(pFilter);
		pIFilterEnum->Reset();
	}

err:
	SAFE_RELEASE(pFilter);
	SAFE_RELEASE(pIFilterEnum);

	return hr;
}

//////////////////////////////////////////////////////////////////////////
// play a video file to our virtual camera

HRESULT CVCamSDKImpl::PlayVideoFile(LPCTSTR fileName)
{
	StopPlay();

	// create filter graph
	HRESULT hr = CoCreateInstance(CLSID_FilterGraph, NULL, 
		CLSCTX_INPROC, IID_IGraphBuilder, (void **)&m_pGraph);
	
	hr = m_pGraph->AddFilter(m_pVCamRenderer, L"Renderer");
	hr = m_pGraph->RenderFile(CT2W(fileName), NULL);
	if (FAILED(hr)) 
		return hr;

	m_pMC = m_pGraph;
	if (NULL != m_pMC) m_pMC->Run();
	m_pMS = m_pGraph;

	// -
	m_bStopGraphCheck = FALSE;
	DWORD threadId;
	m_hGraphCheckThread = ::CreateThread(NULL, 0, GraphCheckThread, this, 0, &threadId);
	return S_OK;
}


LONG	CVCamSDKImpl::GetCurrentVideoDuration()
{
	if ((m_pGraph != NULL) && (m_pMS != NULL)) {
		REFERENCE_TIME rtTotalTime = 0;
		if (SUCCEEDED(m_pMS->GetDuration(&rtTotalTime)))
			return (LONG)(rtTotalTime / (UNITS / MILLISECONDS));
	}

	return 0L;
}

LONG	CVCamSDKImpl::GetCurrentVideoPosition()
{
	if ((m_pGraph != NULL) && (m_pMS != NULL)) {
		REFERENCE_TIME timeNow;
		if (SUCCEEDED(m_pMS->GetCurrentPosition(&timeNow)))
			return (LONG)(timeNow / (UNITS / MILLISECONDS));
	}

	return 0L;
}

HRESULT	CVCamSDKImpl::SetCurrentVideoPosition(LONG pos)
{
	if ((m_pGraph != NULL) && (m_pMS != NULL)) {
		REFERENCE_TIME rtPos = pos * (UNITS / MILLISECONDS);
		return m_pMS->SetPositions(&rtPos, AM_SEEKING_AbsolutePositioning, 
			NULL, AM_SEEKING_NoPositioning);
	}

	return E_FAIL;
}


bool CVCamSDKImpl::IsPlaying() 
{ 
	if ((m_pGraph != NULL) && (m_pMC != NULL)) {
		FILTER_STATE fs;
		m_pMC->GetState(100, (OAFilterState *)&fs);
		return (State_Running == fs); 
	}

	return false;
}

//////////////////////////////////////////////////////////////////////////
// play another webcam's video to our virtual camera
HRESULT CVCamSDKImpl::PlayVideoDevice(UINT idx)
{
	StopPlay();

	// enumerate all video capture devices
	ICreateDevEnum*	pCreateDevEnum;
	IEnumMoniker*	pEnum;	
	IMoniker*		pM = NULL;
	ULONG			cFetched;

	// let's get a device enumerator.
	HRESULT hr = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, 
		CLSCTX_INPROC_SERVER, IID_ICreateDevEnum, (void **)&pCreateDevEnum);
	if (FAILED(hr)) return hr;

	// get an enumerator to enumerate video capture devices.
	hr = pCreateDevEnum->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &pEnum, 0);
	if (FAILED(hr)) return hr;

	// reset the enumerators position.
	if (pEnum) {
		hr = pEnum->Reset();

		// while we have devices to grab let's get them.
		int i=0;
		while (pEnum->Next(1, &pM, &cFetched) == S_OK) {			
			IPropertyBagPtr pBag;
			HRESULT hResult = pM->BindToStorage(0, 0, IID_IPropertyBag, (void **)&pBag);
			if (idx == i++)
				break;
			SAFE_RELEASE(pM);
		}
		SAFE_RELEASE(pEnum);
	}
	SAFE_RELEASE(pCreateDevEnum);

	// we found it
	if (pM) {
		ICaptureGraphBuilder2*	pBuilder = NULL;
		IBaseFilter*	pCapSrcFilter = NULL;
		IBindCtx *pbc = NULL; 
		CreateBindCtx(0, &pbc); 
		hr = pM->BindToObject(pbc, 0, IID_IBaseFilter, (void**)&pCapSrcFilter);
		SAFE_RELEASE(pbc);

		// now create the filter graph
		hr = CoCreateInstance(CLSID_CaptureGraphBuilder2, NULL, 
			CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder2, (void**)&pBuilder);

		hr = CoCreateInstance(CLSID_FilterGraph, NULL, 
			CLSCTX_INPROC, IID_IGraphBuilder, (void **)&m_pGraph);

		pBuilder->SetFiltergraph(m_pGraph);

		hr = m_pGraph->AddFilter(pCapSrcFilter, L"Source");
		hr = m_pGraph->AddFilter(m_pVCamRenderer, L"Renderer");
		hr = pBuilder->RenderStream(NULL, NULL, pCapSrcFilter, NULL, m_pVCamRenderer);
		if (FAILED(hr)) return hr;

		m_pMC = m_pGraph;
		if (NULL != m_pMC) m_pMC->Run();
		m_pMS = m_pGraph;

		SAFE_RELEASE(pCapSrcFilter);
		SAFE_RELEASE(pM);
		SAFE_RELEASE(pBuilder);
	}
	return S_OK;
}

// stop video file or device play
void	CVCamSDKImpl::StopPlay()
{
	if (NULL != m_pGraph) {
		m_pMC->Stop();

		// -
		if (m_hGraphCheckThread) {
			m_bStopGraphCheck = TRUE;
			if (WAIT_TIMEOUT == WaitForSingleObject(m_hGraphCheckThread, 100))
				TerminateThread(m_hGraphCheckThread, 0);
			CloseHandle(m_hGraphCheckThread);
			m_hGraphCheckThread = NULL;
		}

		TearDownGraph(m_pGraph);
		m_pGraph = NULL;
		m_pMC = NULL;
		m_pMS = NULL;
	}
}

void	CVCamSDKImpl::CaptureScreen()
{
	StopPlay();

	if (m_pIVCamSource) {
		m_pIVCamSource->CaptureScreen();
	}
}

void	CVCamSDKImpl::CaptureScreen(RECT rcZone)
{
	StopPlay();

	if (m_pIVCamSource) {
		m_pIVCamSource->CaptureScreenFixed(rcZone.left, rcZone.top, rcZone.right - rcZone.left, rcZone.bottom - rcZone.top);
	}
}

void	CVCamSDKImpl::StopCapture()
{
	if (m_pIVCamSource) {
		m_pIVCamSource->StopCaptureScreen();
	}
}

// set RGB24 image buffer as current virtual camera output
void	CVCamSDKImpl::PlayBuffer(PBYTE rgb24Buffer, DWORD dwWidth, DWORD dwHeight)
{
	if (m_pIVCamRenderer) {
		VARIANT var;
		VariantInit(&var);
		SAFEARRAY *psa;
		SAFEARRAYBOUND rgsabound[1];	

		int buf_size = ((dwWidth*24+31)/32*4)*dwHeight;

		rgsabound[0].lLbound = 0;
		rgsabound[0].cElements = buf_size;
		psa = SafeArrayCreate(VT_UI1, 1, rgsabound);
		BYTE *pBits;
		SafeArrayAccessData(psa, (void **)&pBits);

		memcpy(pBits, rgb24Buffer, buf_size);

		SafeArrayUnaccessData(psa);

		var.vt = VT_UI1|VT_ARRAY;
		var.parray = psa;
		m_pIVCamRenderer->PlayBuffer24(var, dwWidth, dwHeight);
		SafeArrayDestroy(psa);
	}
}

void	CVCamSDKImpl::AddOverlay(BYTE *pRgb32Buffer, long width, long height, long px, long py, long whichOne)
{
	if (m_pIVCamRenderer) {
		VARIANT var;
		VariantInit(&var);
		SAFEARRAY *psa;
		SAFEARRAYBOUND rgsabound[1];	

		int buf_size = width * height * 4;

		rgsabound[0].lLbound = 0;
		rgsabound[0].cElements = buf_size;
		psa = SafeArrayCreate(VT_UI1, 1, rgsabound);
		BYTE *pBits;
		SafeArrayAccessData(psa, (void **)&pBits);

		memcpy(pBits, pRgb32Buffer, buf_size);

		SafeArrayUnaccessData(psa);

		var.vt = VT_UI1|VT_ARRAY;
		var.parray = psa;
		m_pIVCamRenderer->AddOverlay(var, width, height, px, py, whichOne);
	}
}

void	CVCamSDKImpl::AddText(BSTR olText, BSTR ftName, LONG ftSize, LONG ftStyle, LONG ftColor, LONG bkColor, LONG px, LONG py, LONG bkType, LONG whichOne)
{
	if (m_pIVCamRenderer) {
		m_pIVCamRenderer->AddText(olText, ftName, ftSize, ftStyle, ftColor, bkColor, px, py, bkType, whichOne);
	}
}
