﻿using DirectShowLib;
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;

namespace VCamWindowsForms
{
    public struct Device
    {
        public IMoniker m_moniker;
        public string m_name;

        public override string ToString()
        {
            return m_name;
        }

        public IBaseFilter CreateDevice()
        {
            object source = null;
            Guid iid = typeof(IBaseFilter).GUID;
            m_moniker.BindToObject(null, null, ref iid, out source);
            return (IBaseFilter)source;
        }
    }

    class DSPlayer
    {
        public  List<Device>    m_devices = new List<Device>();
        public  List<Device>    m_vcams = new List<Device>();
        private IGraphBuilder   m_graph_builder;
        private IMediaControl   m_control;
        private IMediaPosition  m_position;
        private IMediaSeeking   m_seek;
        private double          m_duration = 0;

        //
        // Get all Video Input Devices except VCam
        //
        public void EnumDevices()
        {
            m_devices.Clear();
            m_vcams.Clear();

            int hr = 0; // S_OK

            IEnumMoniker enum_moniker = null;
            IMoniker[] moniker = new IMoniker[1];

            // Create the system device enumerator
            ICreateDevEnum dev_enum = (ICreateDevEnum)new CreateDevEnum();

            // Create an enumerator for the video capture devices
            hr = dev_enum.CreateClassEnumerator(FilterCategory.VideoInputDevice, out enum_moniker, 0);

            // The device enumerator is no more needed
            Marshal.ReleaseComObject(dev_enum);

            // Get all Device
            while (enum_moniker.Next(moniker.Length, moniker, IntPtr.Zero) == 0)
            {
                Device device = new Device();

                // Get IPropertyBag object
                object obj_property = null;
                Guid guid_property = typeof(IPropertyBag).GUID;
                moniker[0].BindToStorage(null, null, ref guid_property, out obj_property);

                // Convert to IPropertyBag interface
                IPropertyBag i_property = (IPropertyBag)obj_property;

                object obj_name = null;
                i_property.Read("FriendlyName", out obj_name, null);
                device.m_name = obj_name as string;
                device.m_moniker = moniker[0];
                moniker[0] = null;

                // Release COM object
                Marshal.ReleaseComObject(i_property);

                if (!IsVCamDevice(device))
                    m_devices.Add(device);
                else
                    m_vcams.Add(device);
            }

            // Release COM objects
            if (moniker[0] != null)
                Marshal.ReleaseComObject(moniker[0]);

            Marshal.ReleaseComObject(enum_moniker);
        }

        //
        // Check if the input device is VCam, we don't list VCam Camera in the list
        //
        private bool IsVCamDevice(Device i_device)
        {
            Guid vcamsdk_property_set_iid = new Guid("BC919FA0-33B3-40FB-BDA3-621DECF9EE33");
            KSPropertySupport feature = KSPropertySupport.Set;
            Guid guid_filter = typeof(IBaseFilter).GUID;
            object obj_filter = null;

            i_device.m_moniker.BindToObject(null, null, ref guid_filter, out obj_filter);
            IKsPropertySet property_set = (IKsPropertySet)obj_filter;
            int hr = property_set.QuerySupported(vcamsdk_property_set_iid, 0, out feature);
            Marshal.ReleaseComObject(property_set);

            return hr == 0 ? true : false;
        }

        //
        // Open camera device with DirectShow.NET
        //
        public bool OpenCamera(Device i_device, IBaseFilter i_vcam_filter)
        {
            // Stop and release interfaces
            Cleanup();

            bool succeeded = true;
            IPin pin_out = null;
            IPin pin_in = null;
            int hr = 0;

            // Create an instance of FilterGraph interface
            m_graph_builder = (IGraphBuilder)new FilterGraph();

            // Add camera source filter to our graph.
            IBaseFilter filter_source = i_device.CreateDevice();
            if (0 != (hr = m_graph_builder.AddFilter(filter_source, "Source Filter")))
            {
                succeeded = false;
                goto exit;
            }

            // Add VCam Render filter to graph, the VCam Render will pass video frames to VCam
            if (0 != (hr = m_graph_builder.AddFilter(i_vcam_filter, "VCam Renderer Filter")))
            {
                succeeded = false;
                goto exit;
            }

            pin_out = DsFindPin.ByDirection(filter_source, PinDirection.Output, 0);
            pin_in = DsFindPin.ByDirection(i_vcam_filter, PinDirection.Input, 0);
            if (pin_out == null || pin_in == null)
            {
                succeeded = false;
                goto exit;
            }

            if (0 != (hr = m_graph_builder.Connect(pin_out, pin_in)))
            {
                succeeded = false;
                goto exit;
            }

            m_control = (IMediaControl)m_graph_builder;
            
        exit:
            if (filter_source != null) Marshal.ReleaseComObject(filter_source);
            if (pin_out != null) Marshal.ReleaseComObject(pin_out);
            if (pin_in != null) Marshal.ReleaseComObject(pin_in);

            return succeeded;            
        }

        public bool OpenFile(string i_file_name, IBaseFilter i_vcam_filter)
        {
            // Stop and release interfaces
            Cleanup();

            Guid guid_lav_decoder = new Guid("{EE30215D-164F-4A92-A4EB-9D4C13390F9F}");
            Guid guid_lav_split = new Guid("{171252A0-8820-4AFE-9DF8-5C92B2D66B04}");
            IBaseFilter lav_decoder_filter = null;
            IBaseFilter lav_split_filter = null;
            IBaseFilter filter_source = null;

            int succeeded_pin_num = 0;
            int hr = 0;

            // Create an instance of FilterGraph interface
            m_graph_builder = (IGraphBuilder)new FilterGraph();

            // Add source filter with File Name to our graph.            
            if (0 != (hr = m_graph_builder.AddSourceFilter(i_file_name, null, out filter_source)))
                goto exit;

            // LAV video decoder is preferred
            try
            {
                lav_decoder_filter = (IBaseFilter)Activator.CreateInstance(Type.GetTypeFromCLSID(guid_lav_decoder));
                lav_split_filter = (IBaseFilter)Activator.CreateInstance(Type.GetTypeFromCLSID(guid_lav_split));
                m_graph_builder.AddFilter(lav_decoder_filter, "LAV Decoder Filter");
                m_graph_builder.AddFilter(lav_split_filter, "LAV splitter Filter");
            }
            catch (System.Runtime.InteropServices.COMException)
            {
            }

            // Add VCam Render filter to graph, the VCam Render will pass video frames to VCam
            if (0 != (hr = m_graph_builder.AddFilter(i_vcam_filter, "VCam Renderer Filter")))
                goto exit;

            // For each output pins of Source Filter, try to connect it
            
            int idx_output_pin = 0;
            while (true)
            {
                IPin pin_out = DsFindPin.ByDirection(filter_source, PinDirection.Output, idx_output_pin);
                if (pin_out == null)
                    break;

                if (0 == m_graph_builder.Render(pin_out))
                    succeeded_pin_num++;

                Marshal.ReleaseComObject(pin_out);
                idx_output_pin++;
            }

            if (lav_decoder_filter != null)
            {
                if (!IsFilterConnected(lav_decoder_filter))
                    m_graph_builder.RemoveFilter(lav_decoder_filter);

                Marshal.ReleaseComObject(lav_decoder_filter);
            }

            if (lav_split_filter != null)
            {
                if (!IsFilterConnected(lav_split_filter))
                    m_graph_builder.RemoveFilter(lav_split_filter);

                Marshal.ReleaseComObject(lav_split_filter);
            }

            m_control  = (IMediaControl) m_graph_builder;
            m_position = (IMediaPosition)m_graph_builder;
            m_seek     = (IMediaSeeking) m_graph_builder;

            if (m_position != null)
                m_position.get_Duration(out m_duration);

        exit:
            Marshal.ReleaseComObject(filter_source);
            return succeeded_pin_num > 0;
        }

        private bool IsFilterConnected(IBaseFilter i_filter)
        {
            bool b_connected = false;
            IEnumPins pins;
            i_filter.EnumPins(out pins);
            pins.Reset();

            IPin[] pin = new IPin[1];
            try
            {
                while (pins.Next(1, pin, IntPtr.Zero) == 0)
                {
                    IPin pin_connecting = null;
                    try
                    {
                        pin[0].ConnectedTo(out pin_connecting);
                    }
                    finally
                    {
                        if (pin_connecting != null)
                        {
                            Marshal.ReleaseComObject(pin_connecting);
                            b_connected = true;
                        }
                        if (pin[0] != null) Marshal.ReleaseComObject(pin[0]);
                    }

                    if (b_connected)
                    {
                        break;
                    }
                }
            }
            finally
            {
                Marshal.ReleaseComObject(pins);
            }

            return b_connected;
        }

        public void Cleanup()
        {
            // Stop previewing data
            if (m_control != null)
                m_control.StopWhenReady();

            // Release DirectShow interfaces
            if (m_control != null)
                Marshal.ReleaseComObject(m_control); m_control = null;

            if (m_position != null)
                Marshal.ReleaseComObject(m_position); m_position = null;

            if (m_seek != null)
                Marshal.ReleaseComObject(m_seek); m_seek = null;

            if (m_graph_builder != null)
                Marshal.ReleaseComObject(m_graph_builder); m_graph_builder = null;

            m_duration = 0;
        }

        public void Run()
        {
            if (m_control != null)
                m_control.Run();
        }

        public void Stop()
        {
            if (m_control != null)
                m_control.Stop();
        }

        public void Pause()
        {
            if (m_control != null)
                m_control.Pause();
        }

        public double GetDuration()
        {
            return m_duration;
        }

        public void Seek(int second)
        {
            if (m_position != null)
                m_position.put_CurrentPosition(second);
        }

        public double GetCurrentPosition()
        {
            if (m_seek != null)
            {
                long current_position = 0;
                m_seek.GetCurrentPosition(out current_position);
                
                // convert to seconds
                current_position /= 10000;
                return current_position;
            }
            return 0;
        }
    }
}
