﻿using System;
using System.Drawing;
using System.Windows.Forms;
using DirectShowLib;
using System.Threading;

namespace VCamWindowsForms
{
    public partial class Form : System.Windows.Forms.Form
    {
        private DSPlayer                        m_player = new DSPlayer();      // It uses DirectShow.NET to play camera or video file
        private VCamRendererLib.IVCamRenderer   m_vcam;                         // VCam instance
        private IBaseFilter                     m_vcam_filter;                  // VCam Renderer DirectShow Filter
        private System.Windows.Forms.Timer      m_timer;
        private int                             m_cached_pixels_capacity = 0;   // Buffer capacity
        private byte[]                          m_cached_pixels;                // Buffer used to transfer image to VCam
        AutoResetEvent                          m_event;
        private bool                            m_b_wait_event = false;
        SynchronizationContext                  m_sync_context = null;
        private int                             m_current_vcam_idx = 0;
        private Thread                          m_thread = null;

        public Form()
        {
            InitializeComponent();
        }

        private void Form_Load(object sender, EventArgs e)
        {
            FormBorderStyle = FormBorderStyle.FixedSingle;

            // a timer to update video playing progress
            m_timer = new System.Windows.Forms.Timer();
            m_timer.Tick += new EventHandler(Timer_Tick);   // Everytime timer ticks, timer_Tick will be called
            m_timer.Interval = 1000;                        // Timer will tick evert 1 second

            // three filling mode are supported: Uniform, Uniform Fill and Stretch
            comboBoxFillMode.Items.Add("Aspect Fit");
            comboBoxFillMode.Items.Add("Aspect Fill");
            comboBoxFillMode.Items.Add("Stretch");
            comboBoxFillMode.SelectedIndex = 0;

            // fill output format combobox
            comboBoxOutputFormat.Items.Add(" 320 x 240");
            comboBoxOutputFormat.Items.Add(" 640 x 360");
            comboBoxOutputFormat.Items.Add(" 640 x 480");
            comboBoxOutputFormat.Items.Add("1280 x 720");
            comboBoxOutputFormat.Items.Add("1920 x 1080");
            comboBoxOutputFormat.SelectedIndex = 2;

            // get all camera input devices except for VCam
            EnumCameraDevices();

            // Create an instance of VCam Render filter to graph, the VCam Render will pass video frames to VCam
            Guid guid_vcam = new Guid("{3D2F839E-1186-4FCE-B772-B61FAE1ACED7}");
            try
            {
                m_vcam_filter = (IBaseFilter)Activator.CreateInstance(Type.GetTypeFromCLSID(guid_vcam));
                m_vcam = (VCamRendererLib.IVCamRenderer)m_vcam_filter;
            }
            catch (System.Runtime.InteropServices.COMException)
            {
                MessageBox.Show("VCam Renderer Filter not registered!", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                Application.Exit();
                return;
            }
            catch (InvalidCastException)
            {
                MessageBox.Show("VCam Driver not installed!", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                Application.Exit();
                return;
            }

            // which fill mode is using now?
            int current_fill_mode = -1;
            m_vcam.GetFillMode(out current_fill_mode);
            comboBoxFillMode.SelectedIndex = current_fill_mode;

            DetectVCamUsage();
        }

        private void Timer_Tick(object sender, EventArgs e)
        {
            double milliseconds = m_player.GetCurrentPosition();
            double seconds = milliseconds / 1000;
            SetCurrentTime(seconds, m_player.GetDuration());
            if (trackBarPosition.Maximum < (int)seconds)
                trackBarPosition.SetRange(0, (int)seconds);
            trackBarPosition.Value = (int)seconds;
        }

        private void EnumCameraDevices()
        {
            comboBoxDevices.Items.Clear();

            // Get all available Video Input Devices
            m_player.EnumDevices();
            foreach (var device in m_player.m_devices)
            {
                comboBoxDevices.Items.Add(device);
            }
            if (comboBoxDevices.Items.Count > 0)
                comboBoxDevices.SelectedIndex = 0;

            // Get multiple VCam instances
            foreach (var device in m_player.m_vcams)
            {
                comboBoxVCams.Items.Add(device);
            }
            if (comboBoxVCams.Items.Count > 0)
            {
                comboBoxVCams.SelectedIndex = 0;
                textBoxFriendlyName.Text = comboBoxVCams.Items[0].ToString();
            }

            comboBoxVCams.Enabled = (m_player.m_vcams.Count > 1) ? true : false;
        }

        private void StopPlaying()
        {
            m_timer.Stop();
            m_player.Stop();

            buttonTakePhoto.Enabled = false;
            buttonStopVideoDevice.Enabled = false;
            buttonStopVideo.Enabled = false;

            labelTime.Visible = false;
            trackBarPosition.Enabled = false;
            trackBarPosition.Value = 0;
        }

        private void comboBoxVCams_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (comboBoxVCams.SelectedIndex == m_current_vcam_idx)
                return;

            // When set current VCam Device, make sure there is no video file or camera is playing
            StopPlaying();

            Thread.Sleep(1000);

            ////////////////////////////////////////////////////////////////
            // quit previous vcam-using detect thread
            m_b_wait_event = false;
            m_vcam.SetConnectionNotificationEvent(0);
            m_thread.Join();

            // set VCam instance which we want to use
            var device = (VCamWindowsForms.Device)comboBoxVCams.Items[comboBoxVCams.SelectedIndex];
            m_vcam.SetCurrentDevice(device.m_name);
            textBoxFriendlyName.Text = device.m_name;

            // create a new thread to detect VCam using
            DetectVCamUsage();
        }

        private void buttonVideoDevice_Click(object sender, EventArgs e)
        {
            if (comboBoxDevices.SelectedIndex < 0)
                return;

            StopPlaying();

            Device device = (Device)comboBoxDevices.Items[comboBoxDevices.SelectedIndex];
            if (m_player.OpenCamera(device, m_vcam_filter))
            {
                buttonStopVideoDevice.Enabled = true;
                buttonTakePhoto.Enabled = true;
                m_player.Run();
            }
        }

        private void buttonStopVideoDevice_Click(object sender, EventArgs e)
        {
            StopPlaying();
        }

        private void buttonVideoFile_Click(object sender, EventArgs e)
        {
            StopPlaying();

            OpenFileDialog open = new OpenFileDialog();
            open.Filter = "Video files|*.mp4;*.mkv;*.mov;*.rmvb;*.mpg;*.avi|All files(*.*)|*.*";

            if (open.ShowDialog() == DialogResult.OK)
            {
                textBoxVideoFileName.Text = open.SafeFileName;

                // play video file
                if (m_player.OpenFile(open.FileName, m_vcam_filter))
                {
                    trackBarPosition.SetRange(0, (int)m_player.GetDuration());
                    trackBarPosition.Enabled = true;                    
                    buttonStopVideo.Enabled = true;
                    buttonTakePhoto.Enabled = true;
                    labelTime.Visible = true;

                    SetCurrentTime(0, m_player.GetDuration());                    
                    m_player.Run();

                    // start timer
                    m_timer.Start();
                }
            }
        }

        private void SetCurrentTime(double i_position, double i_duration)
        {
            int end_all_seconds = (int)(i_duration);
            int end_hour = (int)(end_all_seconds / (3600));
            int end_minutes = (int)((end_all_seconds / 60) - (end_hour * 60));
            int end_seconds = (int)((end_all_seconds) - (end_minutes * 60) - (end_hour * 3600));

            int pos_all_seconds = (int)(i_position);
            int pos_hour = (int)(pos_all_seconds / (3600));
            int pos_minutes = (int)((pos_all_seconds / 60) - (pos_hour * 60));
            int pos_seconds = (int)((pos_all_seconds) - (pos_minutes * 60) - (pos_hour * 3600));

            labelTime.Text = String.Format("{0:D2}:{1:D2}:{2:D2} / {3:D2}:{4:D2}:{5:D2}", pos_hour, pos_minutes, pos_seconds, end_hour, end_minutes, end_seconds);
        }

        private void buttonStopVideo_Click(object sender, EventArgs e)
        {
            // stop video file playing
            StopPlaying();
        }

        private void buttonPictureFile_Click(object sender, EventArgs e)
        {
            OpenFileDialog open = new OpenFileDialog();
            open.Filter = "Image files|*.jpg; *.jpeg; *.png; *.gif; *.bmp";

            if (open.ShowDialog() == DialogResult.OK)
            {
                // Stop video file and device playing
                StopPlaying();

                textBoxImageFileName.Text = open.SafeFileName;

                // get pixels of image with Bitmap
                Bitmap bitmap = new Bitmap(open.FileName);

                var bitmap_data = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height),
                    System.Drawing.Imaging.ImageLockMode.ReadOnly, System.Drawing.Imaging.PixelFormat.Format24bppRgb);

                try
                {
                    // check cached buffer is big enough
                    int data_size = bitmap_data.Stride * bitmap_data.Height;
                    if (data_size > m_cached_pixels_capacity)
                    {
                        m_cached_pixels_capacity = data_size * 2;
                        m_cached_pixels = new byte[m_cached_pixels_capacity];
                    }

                    System.Runtime.InteropServices.Marshal.Copy(bitmap_data.Scan0, m_cached_pixels, 0, bitmap_data.Stride * bitmap_data.Height);

                    // Play buffer method's parameter: data buffer, width, height, stride
                    // if height is negative, the Driver reads pixels from bottom to top
                    //
                    // SendFrame, SendFrame32 and SendFrameEx can be used to send image to VCam
                    //
                    bool b_buffer_direct = true;
                    if (b_buffer_direct)
                        m_vcam.SendFrame(ref m_cached_pixels[0], bitmap.Width, - bitmap.Height, bitmap_data.Stride);
                    else
                        m_vcam.SendFrameEx(m_cached_pixels, bitmap.Width, - bitmap.Height);
                }
                finally
                {
                    bitmap.UnlockBits(bitmap_data);
                }
            }
        }        
        
        private void buttonClear_Click(object sender, EventArgs e)
        {
            // Stop video file and device playing
            StopPlaying();

            // set empty data to clear displaying image
            byte empty_data = 0;
            m_vcam.SendFrame(ref empty_data, 0, 0, 0);
        }
                
        private void buttonScreenCapture_Click(object sender, EventArgs e)
        {
            StopPlaying();

            int x = Convert.ToInt32(textBoxScreenX.Text);
            int y = Convert.ToInt32(textBoxScreenY.Text);
            int w = Convert.ToInt32(textBoxScreenW.Text);
            int h = Convert.ToInt32(textBoxScreenH.Text);

            if (m_vcam != null)
            {
                m_vcam.CaptureScreen(x, y, w, h);
            }
        }

        private void buttonIdleFileName_Click(object sender, EventArgs e)
        {
            string path = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
            string file_name = path + "\\e2eSoft\\VCamSDK";
            
            // create folder
            System.IO.Directory.CreateDirectory(file_name);
            // welcome image saving file name
            file_name += "\\vcam_idle.bmp";

            OpenFileDialog open = new OpenFileDialog();
            open.Filter = "Image files|*.jpg; *.jpeg; *.png; *.bmp";

            if (open.ShowDialog() == DialogResult.OK)
            {
                // get pixels of image with Bitmap
                Bitmap bitmap = new Bitmap(open.FileName);

                textBoxIdleFileName.Text = open.SafeFileName;
                
                // save as RGB24 bitmap format
                bitmap.Save(file_name, System.Drawing.Imaging.ImageFormat.Bmp);

                // update VCam's idle file
                m_vcam.SetIdleFileName(file_name);
            }
            else
            {
                // restore default Idle image
                m_vcam.SetIdleFileName("");
            }
        }

        private void buttonOutput_Click(object sender, EventArgs e)
        {
            // set output format including width, height and fps
            int sel = comboBoxOutputFormat.SelectedIndex;
            if (sel < 0)
                return;

            int width, height;

            switch (sel)
            {
                case 0:
                    width = 320; height = 240;
                    break;
                case 1:
                    width = 640; height = 360;
                    break;
                case 3:
                    width = 1280; height = 720;
                    break;
                case 4:
                    width = 1920; height = 1080;
                    break;
                case 2:
                default:
                    width = 640; height = 480;
                    break;
            }

            // If FPS is negative value, for example: -1, the camera fps is not fixed, it follows coming video frames rate
            var fps = Convert.ToInt32(textBoxFps.Text);
            m_vcam.SetOutputFormat(width, height, fps, 0, 0);
        }

        private void comboBoxFillMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            var idx = comboBoxFillMode.SelectedIndex;

            // change fill mode
            if (m_vcam != null && idx >= 0)
            {
                m_vcam.SetFillMode((int)idx);
            }
        }

        private void buttonLicense_Click(object sender, EventArgs e)
        {
            // set license code to remove trial displaying
            if (m_vcam != null)
            {
                string license_code = textBoxLicense.Text;
                if (license_code.Length > 0)
                    m_vcam.SetLicenseCode(license_code);
            }
        }

        private void buttonFriendlyName_Click(object sender, EventArgs e)
        {
            if (m_vcam != null)
            {
                string friendly_name = textBoxFriendlyName.Text;
                if (friendly_name.Length > 0)
                    m_vcam.SetFriendlyName(friendly_name);
            }
        }

        private void trackBarPosition_Scroll(object sender, EventArgs e)
        {
            int second = trackBarPosition.Value;
            m_player.Seek(second);
        }

        private void buttonTakePhoto_Click(object sender, EventArgs e)
        {
            // save current VCam frame as bitmap file
            string path = Environment.GetFolderPath(Environment.SpecialFolder.MyPictures);
            string file_name = path + "\\snapshot.bmp";

            if (m_vcam != null)
            {
                // take snapshot on next frame
                m_vcam.Snapshot(file_name);

                // open folder and locate saved file
                System.Diagnostics.ProcessStartInfo psi = new System.Diagnostics.ProcessStartInfo("Explorer.exe");
                psi.Arguments = "/e,/select," + file_name;
                System.Diagnostics.Process.Start(psi);
            }
        }

        private void checkBoxMirror_CheckedChanged(object sender, EventArgs e)
        {
            if (m_vcam != null)
                m_vcam.SetMirror(checkBoxMirror.Checked ? 1 : 0);
        }

        private void checkBoxFlip_CheckedChanged(object sender, EventArgs e)
        {
            if (m_vcam != null)
                m_vcam.SetFlip(checkBoxFlip.Checked ? 1 : 0);
        }

        private void checkBoxRotate_CheckedChanged(object sender, EventArgs e)
        {
            if (m_vcam != null)
                m_vcam.SetRotateRight(checkBoxRotate.Checked ? 1 : 0);
        }

        private void buttonQuit_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void DetectVCamUsage()
        {
            DisplayVCamConnectedCount(null);

            // create a event to detect VCam usage
            m_event = new AutoResetEvent(false);
            
            // set event handle to VCam
            long handle = m_event.SafeWaitHandle.DangerousGetHandle().ToInt64();
            m_vcam.SetConnectionNotificationEvent(handle);

            // create a thread to wait for VCam usage changing notification
            m_sync_context = SynchronizationContext.Current;
            m_b_wait_event = true;
            m_thread = new Thread(DetectUsingThread);
            m_thread.Start(m_sync_context);
        }

        private void DisplayVCamConnectedCount(object state)
        {
            if (m_vcam == null)
                return;

            int vcam_connected_count = 0;
            if (m_vcam != null)
                m_vcam.GetConnectedCount(out vcam_connected_count);

            string message = "No application is using VCam.";
            if (vcam_connected_count == 1) message = "1 application is using VCam.";
            else if (vcam_connected_count > 1) message = String.Format("{0:D} applications are using VCam.", vcam_connected_count);

            labelUsage.Text = message;
        }

        void DetectUsingThread(object state)
        {
            if (m_vcam == null)
                return;

            // grab the context from the state
            SynchronizationContext ui_context = state as SynchronizationContext;

            while (m_b_wait_event)
            {
                // wait for the notification event
                m_event.WaitOne();

                // update using information
                if (m_b_wait_event)
                {
                    ui_context.Post(DisplayVCamConnectedCount, null);
                }
            }
        }

        private void Form_FormClosing(object sender, FormClosingEventArgs e)
        {
            // quit VCam detecting thread //////////////////////////////////
            
            // set thread's loop flag to false
            m_b_wait_event = false;

            // set en empty event handle to VCam
            if (m_vcam != null)
                m_vcam.SetConnectionNotificationEvent(0);
            
            // set event
            //m_event.Set();
        }
    }
}
